--
-- CultivatorFieldCreator
--
-- Author: Wopster
-- Description: allows field creation with cultivators.
-- Icon:
-- Name: cultivatorFieldCreator
-- Hide: no
--
-- Copyright (c) Wopster, 2024

---@class CultivatorFieldCreator
CultivatorFieldCreator = {}
CultivatorFieldCreator.MOD_NAME = g_currentModName
CultivatorFieldCreator.PERMISSION_ALLOW_CREATE_FIELDS = "createFields"

--- Checks if prerequisites are present
---@param specializations table
---@return boolean
function CultivatorFieldCreator.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Cultivator, specializations)
end

--- Registers functions for the vehicle type
---@param vehicleType table
function CultivatorFieldCreator.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setLimitToField", CultivatorFieldCreator.setLimitToField)
end

--- Registers event listeners for the vehicle type
---@param vehicleType table
function CultivatorFieldCreator.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", CultivatorFieldCreator)
    SpecializationUtil.registerEventListener(vehicleType, "onPostLoad", CultivatorFieldCreator)
    SpecializationUtil.registerEventListener(vehicleType, "onPostAttach", CultivatorFieldCreator)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", CultivatorFieldCreator)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", CultivatorFieldCreator)
end

--- Registers overwritten functions for the vehicle type
---@param vehicleType table
function CultivatorFieldCreator.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "updateCultivatorEnabledState", CultivatorFieldCreator.updateCultivatorEnabledState)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "processCultivatorArea", CultivatorFieldCreator.processCultivatorArea)
end

--- Called on load
function CultivatorFieldCreator:onLoad()
    self.spec_cultivatorFieldCreator = self[("spec_%s.cultivatorFieldCreator"):format(CultivatorFieldCreator.MOD_NAME)]
end

--- Triggered after the vehicle is loaded
function CultivatorFieldCreator:onPostLoad()
    local spec = self.spec_cultivator
    spec.limitToField = true
    spec.forceLimitToField = false
end

--- Triggered after the implement is attached
function CultivatorFieldCreator:onPostAttach()
    local spec = self.spec_cultivator
    if not spec.limitToField then
        self:setLimitToField(not spec.limitToField)
    end
end

--- Called on update
function CultivatorFieldCreator:onUpdate()
    if self.isClient then
        local spec = self.spec_cultivatorFieldCreator
        local actionEvent = spec.actionEvents[InputAction.TOGGLE_FIELD_CREATION]
        if actionEvent ~= nil then
            local spec_cultivator = self.spec_cultivator
            if not spec_cultivator.forceLimitToField and g_currentMission:getHasPlayerPermission(CultivatorFieldCreator.PERMISSION_ALLOW_CREATE_FIELDS) then
                local text = spec_cultivator.limitToField and "action_allowCreateFields" or "action_limitToFields"
                g_inputBinding:setActionEventActive(actionEvent.actionEventId, true)
                g_inputBinding:setActionEventText(actionEvent.actionEventId, g_i18n:getText(text))
            else
                g_inputBinding:setActionEventActive(actionEvent.actionEventId, false)
            end
        end
    end
end

--- Sets the cultivator limit to the field state
---@param limitToField boolean
---@param noEventSend boolean
function CultivatorFieldCreator:setLimitToField(limitToField, noEventSend)
    local spec_cultivator = self.spec_cultivator

    if spec_cultivator.limitToField ~= limitToField then
        if noEventSend == nil or noEventSend == false then
            if g_server ~= nil then
                g_server:broadcastEvent(LimitToFieldEvent.new(self, limitToField), nil, nil, self)
            else
                g_client:getServerConnection():sendEvent(LimitToFieldEvent.new(self, limitToField))
            end
        end

        spec_cultivator.limitToField = limitToField
        self:updateCultivatorEnabledState()

        local spec = self.spec_cultivatorFieldCreator
        local actionEvent = spec.actionEvents[InputAction.TOGGLE_FIELD_CREATION]
        if actionEvent ~= nil then
            local text = spec_cultivator.limitToField and "action_allowCreateFields" or "action_limitToFields"
            g_inputBinding:setActionEventText(actionEvent.actionEventId, g_i18n:getText(text))
        end
    end
end

--- Enable cultivator when we don't limit to the field for power harrows
---@param superFunc function
function CultivatorFieldCreator:updateCultivatorEnabledState(superFunc)
    local spec_cultivator = self.spec_cultivator
    if not spec_cultivator.limitToField and spec_cultivator.isPowerHarrow then
        spec_cultivator.isEnabled = true
        return
    end

    return superFunc(self)
end

--- Clear the deco layer when the cultivator is not limited to the field
---@param superFunc function
---@param workArea table
---@param dt number
---@return number, number
function CultivatorFieldCreator:processCultivatorArea(superFunc, workArea, dt)
    local spec_cultivator = self.spec_cultivator
    local realArea, area = superFunc(self, workArea, dt)

    if spec_cultivator.isEnabled and not spec_cultivator.limitToField then
        local xs, _, zs = getWorldTranslation(workArea.start)
        local xw, _, zw = getWorldTranslation(workArea.width)
        local xh, _, zh = getWorldTranslation(workArea.height)
        FSDensityMapUtil.clearDecoArea(xs, zs, xw, zw, xh, zh)
    end

    return realArea, area
end

--- Register the limit to field event
---@param isActiveForInput boolean
---@param isActiveForInputIgnoreSelection boolean
function CultivatorFieldCreator:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self.spec_cultivatorFieldCreator
        self:clearActionEventsTable(spec.actionEvents)
        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_FIELD_CREATION, self, CultivatorFieldCreator.actionEventLimitToField, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_NORMAL)
        end
    end
end

--- ActionEvent LimitToField
---@param self table vehicle table
function CultivatorFieldCreator.actionEventLimitToField(self, ...)
    local spec = self.spec_cultivator
    if not spec.forceLimitToField then
        self:setLimitToField(not spec.limitToField)
    end
end
